/* Target-dependent code for Xilinx MicroBlaze.

   Copyright (C) 2009-2022 Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.  */

#include "defs.h"
#include "arch-utils.h"
#include "dis-asm.h"
#include "frame.h"
#include "trad-frame.h"
#include "symtab.h"
#include "value.h"
#include "gdbcmd.h"
#include "breakpoint.h"
#include "inferior.h"
#include "regcache.h"
#include "target.h"
#include "frame-base.h"
#include "frame-unwind.h"
#include "dwarf2/frame.h"
#include "osabi.h"
#include "target-descriptions.h"
#include "opcodes/microblaze-opcm.h"
#include "opcodes/microblaze-dis.h"
#include "microblaze-tdep.h"
#include "remote.h"

#include "features/microblaze-with-stack-protect.c"
#include "features/microblaze64-with-stack-protect.c"
#include "features/microblaze.c"
#include "features/microblaze64.c"

/* Instruction macros used for analyzing the prologue.  */
/* This set of instruction macros need to be changed whenever the
   prologue generated by the compiler could have more instructions or
   different type of instructions.
   This set also needs to be verified if it is complete.  */
#define IS_RETURN(op) (op == rtsd || op == rtid)
#define IS_UPDATE_SP(op, rd, ra) \
  ((op == addik || op == addi) && rd == REG_SP && ra == REG_SP)
#define IS_SPILL_SP(op, rd, ra) \
  ((op == swi || op == sw) && rd == REG_SP && ra == REG_SP)
#define IS_SPILL_REG(op, rd, ra) \
  ((op == swi || op == sw) && rd != REG_SP && ra == REG_SP)
#define IS_ALSO_SPILL_REG(op, rd, ra, rb) \
  ((op == swi || op == sw) && rd != REG_SP && ra == 0 && rb == REG_SP)
#define IS_SETUP_FP(op, ra, rb) \
  ((op == add || op == addik || op == addk) && ra == REG_SP && rb == 0)
#define IS_SPILL_REG_FP(op, rd, ra, fpregnum) \
  ((op == swi || op == sw) && rd != REG_SP && ra == fpregnum && ra != 0)
#define IS_SAVE_HIDDEN_PTR(op, rd, ra, rb) \
  ((op == add || op == addik) && ra == MICROBLAZE_FIRST_ARGREG && rb == 0)

static const char *microblaze_abi_string;

static const char *const microblaze_abi_strings[] = {
  "auto",
  "m64",
};

enum microblaze_abi
microblaze_abi (struct gdbarch *gdbarch)
{
  microblaze_gdbarch_tdep *tdep = (microblaze_gdbarch_tdep *) gdbarch_tdep (gdbarch);
  return tdep->microblaze_abi;
}
/* The registers of the Xilinx microblaze processor.  */

 static struct cmd_list_element *setmicroblazecmdlist = NULL;
 static struct cmd_list_element *showmicroblazecmdlist = NULL;

static void
microblaze_abi_update (const char *ignore_args,
                 int from_tty, struct cmd_list_element *c)
{
  struct gdbarch_info info;

  /* Force the architecture to update, and (if it's a microblaze architecture)
 *      microblaze_gdbarch_init will take care of the rest.  */
//  gdbarch_info_init (&info);
  gdbarch_update_p (info);
}


static enum microblaze_abi
global_microblaze_abi (void)
{
  int i;

  for (i = 0; microblaze_abi_strings[i] != NULL; i++)
    if (microblaze_abi_strings[i] == microblaze_abi_string)
      return (enum microblaze_abi) i;

//  internal_error (__FILE__, __LINE__, _("unknown ABI string"));
}

static void
show_microblaze_abi (struct ui_file *file,
               int from_tty,
               struct cmd_list_element *ignored_cmd,
               const char *ignored_value)
{
      enum microblaze_abi global_abi = global_microblaze_abi ();
      enum microblaze_abi actual_abi = microblaze_abi (target_gdbarch ());
      const char *actual_abi_str = microblaze_abi_strings[actual_abi];

#if 1 
      if (global_abi == MICROBLAZE_ABI_AUTO)
        fprintf_filtered
          (file,
           "The microblaze ABI is set automatically (currently \"%s\").\n",
           actual_abi_str);	
      else if (global_abi == actual_abi)
        fprintf_filtered
          (file,
           "The microblaze ABI is assumed to be \"%s\" (due to user setting).\n",
           actual_abi_str);
      else
        {
#endif
          /* Probably shouldn't happen...  */
          fprintf_filtered (file,
                            "The (auto detected) microblaze ABI \"%s\" is in use "
                            "even though the user setting was \"%s\".\n",
             actual_abi_str, microblaze_abi_strings[global_abi]);
        }
}		   

static void
show_microblaze_command (const char *args, int from_tty)
{
  help_list (showmicroblazecmdlist, "show microblaze ", all_commands, gdb_stdout);
}

static void
set_microblaze_command (const char *args, int from_tty)
{
  printf_unfiltered
    ("\"set microblaze\" must be followed by an appropriate subcommand.\n");
  help_list (setmicroblazecmdlist, "set microblaze ", all_commands, gdb_stdout);
}

static const char * const microblaze_register_names[] =
{
  "r0",   "r1",  "r2",    "r3",   "r4",   "r5",   "r6",   "r7",
  "r8",   "r9",  "r10",   "r11",  "r12",  "r13",  "r14",  "r15",
  "r16",  "r17", "r18",   "r19",  "r20",  "r21",  "r22",  "r23",
  "r24",  "r25", "r26",   "r27",  "r28",  "r29",  "r30",  "r31",
  "rpc",  "rmsr", "rear", "resr", "rfsr", "rbtr",
  "rpvr0", "rpvr1", "rpvr2", "rpvr3", "rpvr4", "rpvr5", "rpvr6",
  "rpvr7", "rpvr8", "rpvr9", "rpvr10", "rpvr11",
  "redr", "rpid", "rzpr", "rtlbx", "rtlbsx", "rtlblo", "rtlbhi",
  "slr", "shr"
};

#define MICROBLAZE_NUM_REGS ARRAY_SIZE (microblaze_register_names)

static unsigned int microblaze_debug_flag = 0;
int reg_size = 4;

unsigned int
microblaze_abi_regsize (struct gdbarch *gdbarch)
{
  switch (microblaze_abi (gdbarch))
    {
    case MICROBLAZE_ABI_M64:
      return 8;
    default:
      return 4;
    }
}

#define microblaze_debug(fmt, ...) \
  debug_prefixed_printf_cond_nofunc (microblaze_debug_flag, "MICROBLAZE", \
                                     fmt, ## __VA_ARGS__)


/* Return the name of register REGNUM.  */

static const char *
microblaze_register_name (struct gdbarch *gdbarch, int regnum)
{
  if (regnum >= 0 && regnum < MICROBLAZE_NUM_REGS)
    return microblaze_register_names[regnum];
  return NULL;
}

static struct type *
microblaze_register_type (struct gdbarch *gdbarch, int regnum)
{
  if (regnum == MICROBLAZE_SP_REGNUM)
    return builtin_type (gdbarch)->builtin_data_ptr;

  if (regnum == MICROBLAZE_PC_REGNUM)
    return builtin_type (gdbarch)->builtin_func_ptr;

  return builtin_type (gdbarch)->builtin_int;
}


/* Fetch the instruction at PC.  */

static unsigned long
microblaze_fetch_instruction (CORE_ADDR pc)
{
  enum bfd_endian byte_order = gdbarch_byte_order (target_gdbarch ());
  gdb_byte buf[4];

  /* If we can't read the instruction at PC, return zero.  */
  if (target_read_code (pc, buf, sizeof (buf)))
    return 0;

  return extract_unsigned_integer (buf, 4, byte_order);
}

constexpr gdb_byte microblaze_break_insn[] = MICROBLAZE_BREAKPOINT;

typedef BP_MANIPULATION (microblaze_break_insn) microblaze_breakpoint;
static CORE_ADDR
microblaze_store_arguments (struct regcache *regcache, int nargs,
			    struct value **args, CORE_ADDR sp,
			    int struct_return, CORE_ADDR struct_addr)
{
  error (_("store_arguments not implemented"));
  return sp;
}
#if 0
static int
microblaze_linux_memory_remove_breakpoint (struct gdbarch *gdbarch,
				    struct bp_target_info *bp_tgt)
{
  CORE_ADDR addr = bp_tgt->placed_address;
  const unsigned char *bp;
  int val;
  int bplen;
  gdb_byte old_contents[BREAKPOINT_MAX];

  /* Determine appropriate breakpoint contents and size for this address.  */
  bp = gdbarch_breakpoint_from_pc (gdbarch, &addr, &bplen);
  if (bp == NULL)
    error (_("Software breakpoints not implemented for this target."));

  /* Make sure we see the memory breakpoints.  */
  scoped_restore restore_memory
    = make_scoped_restore_show_memory_breakpoints (1);
  val = target_read_memory (addr, old_contents, bplen);

  /* If our breakpoint is no longer at the address, this means that the
     program modified the code on us, so it is wrong to put back the
     old value.  */
  if (val == 0 && memcmp (bp, old_contents, bplen) == 0)
  {
    val = target_write_raw_memory (addr, bp_tgt->shadow_contents, bplen);
    microblaze_debug ("microblaze_linux_memory_remove_breakpoint writing back to memory at addr 0x%lx\n", addr);
  }

  return val;
}

#endif
/* Allocate and initialize a frame cache.  */

static struct microblaze_frame_cache *
microblaze_alloc_frame_cache (void)
{
  struct microblaze_frame_cache *cache;

  cache = FRAME_OBSTACK_ZALLOC (struct microblaze_frame_cache);

  /* Base address.  */
  cache->base = 0;
  cache->pc = 0;

  /* Frameless until proven otherwise.  */
  cache->frameless_p = 1;

  return cache;
}

/* The base of the current frame is actually in the stack pointer.
   This happens when there is no frame pointer (microblaze ABI does not
   require a frame pointer) or when we're stopped in the prologue or
   epilogue itself.  In these cases, microblaze_analyze_prologue will need
   to update fi->frame before returning or analyzing the register
   save instructions.  */
#define MICROBLAZE_MY_FRAME_IN_SP 0x1

/* The base of the current frame is in a frame pointer register.
   This register is noted in frame_extra_info->fp_regnum.

   Note that the existance of an FP might also indicate that the
   function has called alloca.  */
#define MICROBLAZE_MY_FRAME_IN_FP 0x2

/* Function prologues on the Xilinx microblaze processors consist of:

   - adjustments to the stack pointer (r1) (addi r1, r1, imm)
   - making a copy of r1 into another register (a "frame" pointer)
     (add r?, r1, r0)
   - store word/multiples that use r1 or the frame pointer as the
     base address (swi r?, r1, imm OR swi r?, fp, imm)

   Note that microblaze really doesn't have a real frame pointer.
   Instead, the compiler may copy the SP into a register (usually
   r19) to act as an arg pointer.  For our target-dependent purposes,
   the frame info's "frame" member will be the beginning of the
   frame.  The SP could, in fact, point below this.

   The prologue ends when an instruction fails to meet either of
   these criteria.  */

/* Analyze the prologue to determine where registers are saved,
   the end of the prologue, etc.  Return the address of the first line
   of "real" code (i.e., the end of the prologue).  */

static CORE_ADDR
microblaze_analyze_prologue (struct gdbarch *gdbarch, CORE_ADDR pc, 
			     CORE_ADDR current_pc,
			     struct microblaze_frame_cache *cache)
{
  const char *name;
  CORE_ADDR func_addr, func_end, addr, stop, prologue_end_addr = 0;
  unsigned long insn;
  int rd, ra, rb, imm;
  enum microblaze_instr op;
  int save_hidden_pointer_found = 0;
  int non_stack_instruction_found = 0;

  /* Find the start of this function.  */
  find_pc_partial_function (pc, &name, &func_addr, &func_end);
  if (func_addr < pc)
    pc = func_addr;

  if (current_pc < pc)
    return current_pc;

   /* Initialize info about frame.  */
   cache->framesize = 0;
   cache->fp_regnum = MICROBLAZE_SP_REGNUM;
   cache->frameless_p = 1;

  /* Start decoding the prologue.  We start by checking two special cases:

     1. We're about to return
     2. We're at the first insn of the prologue.

     If we're about to return, our frame has already been deallocated.
     If we are stopped at the first instruction of a prologue,
     then our frame has not yet been set up.  */

  /* Get the first insn from memory.  */

  insn = microblaze_fetch_instruction (pc);
  op = microblaze_decode_insn (insn, &rd, &ra, &rb, &imm);

  if (IS_RETURN(op))
    return pc;

  /* Start at beginning of function and analyze until we get to the
     current pc, or the end of the function, whichever is first.  */
  stop = (current_pc < func_end ? current_pc : func_end);

  microblaze_debug ("Scanning prologue: name=%s, func_addr=%s, stop=%s\n", 
		    name, paddress (gdbarch, func_addr), 
		    paddress (gdbarch, stop));

  for (addr = func_addr; addr < stop; addr += INST_WORD_SIZE)
    {
      insn = microblaze_fetch_instruction (addr);
      op = microblaze_decode_insn (insn, &rd, &ra, &rb, &imm);
      microblaze_debug ("%s %08lx\n", paddress (gdbarch, pc), insn);

      /* This code is very sensitive to what functions are present in the
	 prologue.  It assumes that the (addi, addik, swi, sw) can be the 
	 only instructions in the prologue.  */
      if (IS_UPDATE_SP(op, rd, ra))
	{
	  microblaze_debug ("got addi r1,r1,%d; contnuing\n", imm);
	  if (cache->framesize)
	    break;	/* break if framesize already computed.  */
	  cache->framesize = -imm; /* stack grows towards low memory.  */
	  cache->frameless_p = 0; /* Frame found.  */
	  save_hidden_pointer_found = 0;
	  non_stack_instruction_found = 0;
          cache->register_offsets[rd] = -imm;
	  continue;
	}
      else if (IS_SPILL_SP(op, rd, ra))
	{
	  /* Spill stack pointer.  */
	  cache->register_offsets[rd] = imm; /* SP spilled before updating.  */

	  microblaze_debug ("swi r1 r1 %d, continuing\n", imm);
	  save_hidden_pointer_found = 0;
	  if (!cache->framesize)
	    non_stack_instruction_found = 0;
	  continue;
	}
      else if (IS_SPILL_REG(op, rd, ra))
	{
	  /* Spill register.  */
	  cache->register_offsets[rd] = imm - cache->framesize;

	  microblaze_debug ("swi %d r1 %d, continuing\n", rd, imm);
	  save_hidden_pointer_found = 0;
	  if (!cache->framesize)
	    non_stack_instruction_found = 0;
	  continue;
	}
      else if (IS_ALSO_SPILL_REG(op, rd, ra, rb))
	{
	  /* Spill register.  */
	  cache->register_offsets[rd] = 0 - cache->framesize;

	  microblaze_debug ("sw %d r0 r1, continuing\n", rd);
	  save_hidden_pointer_found = 0;
	  if (!cache->framesize)
	    non_stack_instruction_found = 0;
	  continue;
	}
      else if (IS_SETUP_FP(op, ra, rb))
	{
	  /* We have a frame pointer.  Note the register which is 
	     acting as the frame pointer.  */
	  cache->fp_regnum = rd;
	  microblaze_debug ("Found a frame pointer: r%d\n", cache->fp_regnum);
	  save_hidden_pointer_found = 0;
	  if (!cache->framesize)
	    non_stack_instruction_found = 0;
	  continue;
	}
      else if (IS_SPILL_REG_FP(op, rd, ra, cache->fp_regnum))
	{
	  /* reg spilled after updating.  */
	  cache->register_offsets[rd] = imm - cache->framesize;

	  microblaze_debug ("swi %d %d %d, continuing\n", rd, ra, imm);
	  save_hidden_pointer_found = 0;
	  if (!cache->framesize)
	    non_stack_instruction_found = 0;
	  continue;
	}
      else if (IS_SAVE_HIDDEN_PTR(op, rd, ra, rb))
	{
	  /* If the first argument is a hidden pointer to the area where the
	     return structure is to be saved, then it is saved as part of the
	     prologue.  */

	  microblaze_debug ("add %d %d %d, continuing\n", rd, ra, rb);
	  save_hidden_pointer_found = 1;
	  if (!cache->framesize)
	    non_stack_instruction_found = 0;
	  continue;
	}

      /* As a result of the modification in the next step where we continue
	 to analyze the prologue till we reach a control flow instruction,
	 we need another variable to store when exactly a non-stack
	 instruction was encountered, which is the current definition
	 of a prologue.  */
      if (!non_stack_instruction_found)
	prologue_end_addr = addr;
      non_stack_instruction_found = 1;

      /* When optimizations are enabled, it is not guaranteed that prologue
	 instructions are not mixed in with other instructions from the
	 program.  Some programs show this behavior at -O2.  This can be
	 avoided by adding -fno-schedule-insns2 switch as of now (edk 8.1)
	 In such cases, we scan the function until we see the first control
	 instruction.  */

      {
	unsigned ctrl_op = (unsigned)insn >> 26;

	/* continue if not control flow (branch, return).  */
	if (ctrl_op != 0x26 && ctrl_op != 0x27 && ctrl_op != 0x2d
	    && ctrl_op != 0x2e && ctrl_op != 0x2f)
	  continue;
	else if (ctrl_op == 0x2c)
	  continue;    /* continue if imm.  */
      }

      /* This is not a prologue insn, so stop here.  */
      microblaze_debug ("insn is not a prologue insn -- ending scan\n");
      break;
    }

  microblaze_debug ("done analyzing prologue\n");
  microblaze_debug ("prologue end = 0x%x\n", (int) addr);

  /* If the last instruction was an add rd, r5, r0 then don't count it as
     part of the prologue.  */
  if (save_hidden_pointer_found)
    prologue_end_addr -= INST_WORD_SIZE;

  return prologue_end_addr;
}

static CORE_ADDR
microblaze_unwind_pc (struct gdbarch *gdbarch, struct frame_info *next_frame)
{
  CORE_ADDR pc;
  pc=frame_unwind_register_unsigned (next_frame, MICROBLAZE_PC_REGNUM);
  return pc;
}

/* Return PC of first real instruction of the function starting at
   START_PC.  */

static CORE_ADDR
microblaze_skip_prologue (struct gdbarch *gdbarch, CORE_ADDR start_pc)
{
  struct symtab_and_line sal;
  CORE_ADDR func_start, func_end, ostart_pc;
  struct microblaze_frame_cache cache;

  /* This is the preferred method, find the end of the prologue by
     using the debugging information.  Debugging info does not always
     give the right answer since parameters are stored on stack after this.
     Always analyze the prologue.  */
  if (find_pc_partial_function (start_pc, NULL, &func_start, &func_end))
    {
      sal = find_pc_line (func_start, 0);
 
      if (sal.line !=0 && sal.end <= func_end  && start_pc <= sal.end) {
	start_pc = sal.end;
    microblaze_debug("start_pc is %d\t sal.end is %d\t func_end is %d\t",start_pc,sal.end,func_end);
    }
    }

  ostart_pc = microblaze_analyze_prologue (gdbarch, func_start, 0xffffffffUL, 
					   &cache);

  if (ostart_pc > start_pc)
    return ostart_pc;
  return start_pc;
}

/* Normal frames.  */

static struct microblaze_frame_cache *
microblaze_frame_cache (struct frame_info *next_frame, void **this_cache)
{
  struct microblaze_frame_cache *cache;
  struct gdbarch *gdbarch = get_frame_arch (next_frame);
  int rn;
  CORE_ADDR current_pc;

  if (*this_cache)
    return (struct microblaze_frame_cache *) *this_cache;

  cache = microblaze_alloc_frame_cache ();
  *this_cache = cache;
  cache->saved_regs = trad_frame_alloc_saved_regs (next_frame);

  /* Clear offsets to saved regs in frame.  */
  for (rn = 0; rn < gdbarch_num_regs (gdbarch); rn++)
    cache->register_offsets[rn] = -1;

  /* Call for side effects.  */
  cache->pc = get_frame_func (next_frame);

//  cache->pc = get_frame_address_in_block (next_frame);
  current_pc = get_frame_pc (next_frame);
  if (cache->pc)
     microblaze_analyze_prologue (gdbarch, cache->pc, current_pc, cache);
  
  cache->saved_sp = cache->base + cache->framesize;
  cache->register_offsets[MICROBLAZE_PREV_PC_REGNUM] = cache->base;
  cache->register_offsets[MICROBLAZE_SP_REGNUM] = cache->saved_sp;
  
  return cache;
}

static void
microblaze_frame_this_id (struct frame_info *next_frame, void **this_cache,
		       struct frame_id *this_id)
{
  struct microblaze_frame_cache *cache =
    microblaze_frame_cache (next_frame, this_cache);

  /* This marks the outermost frame.  */
  if (cache->base == 0)
    return;

  (*this_id) = frame_id_build (cache->base, cache->pc);
}

static struct value *
microblaze_frame_prev_register (struct frame_info *this_frame,
				 void **this_cache, int regnum)
{
  struct microblaze_frame_cache *cache =
    microblaze_frame_cache (this_frame, this_cache);

if ((regnum == MICROBLAZE_SP_REGNUM &&
      cache->register_offsets[MICROBLAZE_SP_REGNUM])
      || (regnum == MICROBLAZE_FP_REGNUM &&
      cache->register_offsets[MICROBLAZE_SP_REGNUM]))

     return frame_unwind_got_constant (this_frame, regnum,
                                       cache->register_offsets[MICROBLAZE_SP_REGNUM]);

if (regnum == MICROBLAZE_PC_REGNUM)
{
      regnum = 15;
      return frame_unwind_got_memory (this_frame, regnum,
                                      cache->register_offsets[MICROBLAZE_PREV_PC_REGNUM]);

}
if (regnum == MICROBLAZE_SP_REGNUM)
        regnum = 1;
#if 0

  if (cache->frameless_p)
    {
      if (regnum == MICROBLAZE_PC_REGNUM)
	regnum = 15;
      if (regnum == MICROBLAZE_SP_REGNUM)
	regnum = 1;
      return trad_frame_get_prev_register (this_frame,
					   cache->saved_regs, regnum);
    }
  else
    return trad_frame_get_prev_register (this_frame, cache->saved_regs,
					 regnum);
#endif
  return trad_frame_get_prev_register (this_frame, cache->saved_regs,
					regnum); 
}

static const struct frame_unwind microblaze_frame_unwind =
{
  "microblaze prologue",
  NORMAL_FRAME,
  default_frame_unwind_stop_reason,
  microblaze_frame_this_id,
  microblaze_frame_prev_register,
  NULL,
  default_frame_sniffer,
  NULL,
};

static CORE_ADDR
microblaze_frame_base_address (struct frame_info *next_frame,
			       void **this_cache)
{
  struct microblaze_frame_cache *cache =
    microblaze_frame_cache (next_frame, this_cache);

  return cache->base;
}

static const struct frame_base microblaze_frame_base =
{
  &microblaze_frame_unwind,
  microblaze_frame_base_address,
  microblaze_frame_base_address,
  microblaze_frame_base_address
};

/* Extract from an array REGBUF containing the (raw) register state, a
   function return value of TYPE, and copy that into VALBUF.  */
static void
microblaze_extract_return_value (struct type *type, struct regcache *regcache,
				 gdb_byte *valbuf)
{
  gdb_byte buf[8];
  /* Copy the return value (starting) in RETVAL_REGNUM to VALBUF.  */
  switch (TYPE_LENGTH (type))
    {
      case 1:	/* return last byte in the register.  */
	regcache->cooked_read (MICROBLAZE_RETVAL_REGNUM, buf);
	memcpy(valbuf, buf + reg_size - 1, 1);
	return;
      case 2:	/* return last 2 bytes in register.  */
	regcache->cooked_read (MICROBLAZE_RETVAL_REGNUM, buf);
	memcpy(valbuf, buf + reg_size - 2, 2);
	return;
      case 4:	/* for sizes 4 or 8, copy the required length.  */
      case 8:
	regcache->cooked_read (MICROBLAZE_RETVAL_REGNUM, buf);
	regcache->cooked_read (MICROBLAZE_RETVAL_REGNUM + 1, buf+4);
	memcpy (valbuf, buf, TYPE_LENGTH (type));
	return;
      default:
	internal_error (__FILE__, __LINE__, 
			_("Unsupported return value size requested"));
    }
}

/* Store the return value in VALBUF (of type TYPE) where the caller
   expects to see it.

   Integers up to four bytes are stored in r3.

   Longs are stored in r3 (most significant word) and r4 (least
   significant word).

   Small structures are always returned on stack.  */

static void
microblaze_store_return_value (struct type *type, struct regcache *regcache,
			       const gdb_byte *valbuf)
{
  int len = TYPE_LENGTH (type);
  gdb_byte buf[8];

  memset (buf, 0, sizeof(buf));

  /* Integral and pointer return values.  */

  if (len > 4)
    {
       gdb_assert (len == 8);
       memcpy (buf, valbuf, 8);
       regcache->cooked_write (MICROBLAZE_RETVAL_REGNUM+1, buf + 4);
    }
  else
    /* ??? Do we need to do any sign-extension here?  */
    memcpy (buf + 4 - len, valbuf, len);

  regcache->cooked_write (MICROBLAZE_RETVAL_REGNUM, buf);
}

static enum return_value_convention
microblaze_return_value (struct gdbarch *gdbarch, struct value *function,
			 struct type *type, struct regcache *regcache,
			 gdb_byte *readbuf, const gdb_byte *writebuf)
{
  if (readbuf)
    microblaze_extract_return_value (type, regcache, readbuf);
  if (writebuf)
    microblaze_store_return_value (type, regcache, writebuf);

  return RETURN_VALUE_REGISTER_CONVENTION;
}

static int
microblaze_stabs_argument_has_addr (struct gdbarch *gdbarch, struct type *type)
{
  return (TYPE_LENGTH (type) == 16);
}

#if 1 
static std::vector<CORE_ADDR>
microblaze_software_single_step (struct regcache *regcache)
{
  struct gdbarch *arch = regcache->arch ();
  //struct gdbarch_tdep *tdep = gdbarch_tdep (arch);
  static int le_breakp[] = MICROBLAZE_BREAKPOINT_LE;
  static int be_breakp[] = MICROBLAZE_BREAKPOINT;
  enum bfd_endian byte_order = gdbarch_byte_order (arch);
  int *breakp = byte_order == BFD_ENDIAN_BIG ? be_breakp : le_breakp;
//  std::vector<CORE_ADDR> ret = NULL;

 /* Save the address and the values of the next_pc and the target */
  static struct sstep_breaks
  {
    CORE_ADDR address;
    bfd_boolean valid;
    /* Shadow contents.  */
    char data[INST_WORD_SIZE];
  } stepbreaks[2];
 int ii;

      CORE_ADDR pc;
      std::vector<CORE_ADDR> next_pcs;
      long insn;
      enum microblaze_instr minstr;
      bfd_boolean isunsignednum;
      enum microblaze_instr_type insn_type;
      short delay_slots;
      int imm;
      bfd_boolean immfound = FALSE;

     /* Set a breakpoint at the next instruction */
      /* If the current instruction is an imm, set it at the inst after */
      /* If the instruction has a delay slot, skip the delay slot */
      pc = regcache_read_pc (regcache);
      insn = microblaze_fetch_instruction (pc);
      minstr = get_insn_microblaze (insn, &isunsignednum, &insn_type, &delay_slots);
      if (insn_type == immediate_inst)
	{
	  int rd, ra, rb;
	  immfound = TRUE;
	  minstr = microblaze_decode_insn (insn, &rd, &ra, &rb, &imm);
	  pc = pc + INST_WORD_SIZE;
	  insn = microblaze_fetch_instruction (pc);
	  minstr = get_insn_microblaze (insn, &isunsignednum, &insn_type, &delay_slots);
	}
      stepbreaks[0].address = pc + (delay_slots * INST_WORD_SIZE) + INST_WORD_SIZE;
      if (insn_type != return_inst) {
	stepbreaks[0].valid = TRUE;
      } else {
	stepbreaks[0].valid = FALSE;
      }

      microblaze_debug ("single-step insn_type=%x insn=%x\n", insn_type, insn);
      /* Now check for branch or return instructions */
      if (insn_type == branch_inst || insn_type == return_inst) {
	int limm;
	int lrd, lra, lrb;
	int ra, rb;
	bfd_boolean targetvalid;
	bfd_boolean unconditionalbranch;
	microblaze_decode_insn(insn, &lrd, &lra, &lrb, &limm);
	if (lra >= 0 && lra < MICROBLAZE_NUM_REGS)
	  ra = regcache_raw_get_unsigned(regcache, lra);
	else
	  ra = 0;
	if (lrb >= 0 && lrb < MICROBLAZE_NUM_REGS)
	  rb = regcache_raw_get_unsigned(regcache, lrb);
	else
	  rb = 0;
	stepbreaks[1].address = microblaze_get_target_address (insn, immfound, imm, pc, ra, rb, &targetvalid, &unconditionalbranch);
        microblaze_debug ("single-step uncondbr=%d targetvalid=%d target=%x\n", unconditionalbranch, targetvalid, stepbreaks[1].address);
	if (unconditionalbranch)
	  stepbreaks[0].valid = FALSE; /* This is a unconditional branch: will not come to the next address */
	if (targetvalid && (stepbreaks[0].valid == FALSE ||
			    (stepbreaks[0].address != stepbreaks[1].address))
	                && (stepbreaks[1].address != pc)) {
	  stepbreaks[1].valid = TRUE;
	} else {
	  stepbreaks[1].valid = FALSE;
	}
      } else {
	stepbreaks[1].valid = FALSE;
      }

      /* Insert the breakpoints */
      for (ii = 0; ii < 2; ++ii)
        {

          /* ignore invalid breakpoint. */
          if (stepbreaks[ii].valid) {
           // VEC_safe_push (CORE_ADDR, next_pcs, stepbreaks[ii].address);;
            next_pcs.push_back (stepbreaks[ii].address);
	  }
	}
    return next_pcs;
}
#endif

static void
microblaze_write_pc (struct regcache *regcache, CORE_ADDR pc)
{
  regcache_cooked_write_unsigned (regcache, MICROBLAZE_PC_REGNUM, pc);
}

static int dwarf2_to_reg_map[78] =
{ 0  /* r0  */,   1  /* r1  */,   2  /* r2  */,   3  /* r3  */,  /*  0- 3 */
  4  /* r4  */,   5  /* r5  */,   6  /* r6  */,   7  /* r7  */,  /*  4- 7 */
  8  /* r8  */,   9  /* r9  */,  10  /* r10 */,  11  /* r11 */,  /*  8-11 */
  12 /* r12 */,  13  /* r13 */,  14  /* r14 */,  15  /* r15 */,  /* 12-15 */
  16 /* r16 */,  17  /* r17 */,  18  /* r18 */,  19  /* r19 */,  /* 16-19 */
  20 /* r20 */,  21  /* r21 */,  22  /* r22 */,  23  /* r23 */,  /* 20-23 */
  24 /* r24 */,  25  /* r25 */,  26  /* r26 */,  27  /* r27 */,  /* 24-25 */
  28 /* r28 */,  29  /* r29 */,  30  /* r30 */,  31  /* r31 */,  /* 28-31 */
  -1 /* $f0 */,  -1  /* $f1 */,  -1  /* $f2 */,  -1  /* $f3 */,  /* 32-35 */
  -1 /* $f4 */,  -1  /* $f5 */,  -1  /* $f6 */,  -1  /* $f7 */,  /* 36-39 */
  -1 /* $f8 */,  -1  /* $f9 */,  -1  /* $f10 */, -1  /* $f11 */, /* 40-43 */
  -1 /* $f12 */, -1  /* $f13 */, -1  /* $f14 */, -1  /* $f15 */, /* 44-47 */
  -1 /* $f16 */, -1  /* $f17 */, -1  /* $f18 */, -1  /* $f19 */, /* 48-51 */
  -1 /* $f20 */, -1  /* $f21 */, -1  /* $f22 */, -1  /* $f23 */, /* 52-55 */
  -1 /* $f24 */, -1  /* $f25 */, -1  /* $f26 */, -1  /* $f27 */, /* 56-59 */
  -1 /* $f28 */, -1  /* $f29 */, -1  /* $f30 */, -1  /* $f31 */, /* 60-63 */
  -1 /* hi   */, -1  /* lo   */, -1  /* accum*/, 33  /* rmsr */, /* 64-67 */
  -1 /* $fcc1*/, -1  /* $fcc2*/, -1  /* $fcc3*/, -1  /* $fcc4*/, /* 68-71 */
  -1 /* $fcc5*/, -1  /* $fcc6*/, -1  /* $fcc7*/, -1  /* $ap  */, /* 72-75 */
  -1 /* $rap */, -1  /* $frp */					 /* 76-77 */
};

static int
microblaze_dwarf2_reg_to_regnum (struct gdbarch *gdbarch, int reg)
{
  if (reg >= 0 && reg < sizeof (dwarf2_to_reg_map))
    return dwarf2_to_reg_map[reg];
  return -1;
}

static void
microblaze_register_g_packet_guesses (struct gdbarch *gdbarch)
{

  register_remote_g_packet_guess (gdbarch,
				  4 * MICROBLAZE_NUM_CORE_REGS,
				  tdesc_microblaze64);

  register_remote_g_packet_guess (gdbarch,
				  4 * MICROBLAZE_NUM_REGS,
				  tdesc_microblaze64_with_stack_protect);
}

void
microblaze_supply_gregset (const struct regset *regset,
                        struct regcache *regcache,
                        int regnum, const void *gregs)
{
  const gdb_byte *regs = (const gdb_byte *) gregs;
  if (regnum >= 0)
    regcache->raw_supply (regnum, regs + regnum);

  if (regnum == -1) {
    int i;

    for (i = 0; i < 50; i++) {
      regcache->raw_supply (regnum, regs + i);
    }
  }
}


/* Return the appropriate register set for the core section identified
   by SECT_NAME and SECT_SIZE.  */

static void
microblaze_iterate_over_regset_sections (struct gdbarch *gdbarch,
                                     iterate_over_regset_sections_cb *cb,
                                     void *cb_data,
                                     const struct regcache *regcache)
{
  struct microblaze_gdbarch_tdep *tdep =(microblaze_gdbarch_tdep *) gdbarch_tdep (gdbarch);

  cb(".reg", tdep->sizeof_gregset, tdep->sizeof_gregset, tdep->gregset, NULL, cb_data);

  cb(".reg2", tdep->sizeof_fpregset, tdep->sizeof_fpregset, tdep->fpregset, NULL, cb_data);
}


static void
make_regs (struct gdbarch *arch)
{
  struct gdbarch_tdep *tdep = gdbarch_tdep (arch);
  int mach = gdbarch_bfd_arch_info (arch)->mach;
  
  if (mach == bfd_mach_microblaze64)
    {
      set_gdbarch_ptr_bit (arch, 64);
    }
}

static struct gdbarch *
microblaze_gdbarch_init (struct gdbarch_info info, struct gdbarch_list *arches)
{
  struct gdbarch *gdbarch;
  tdesc_arch_data_up tdesc_data;
  enum microblaze_abi microblaze_abi, found_abi, wanted_abi;
  const struct target_desc *tdesc = info.target_desc;

 /* What has the user specified from the command line?  */
  wanted_abi = global_microblaze_abi ();
  if (gdbarch_debug)
    fprintf_unfiltered (gdb_stdlog, "microblaze_gdbarch_init: wanted_abi = %d\n",
                        wanted_abi);
  if (wanted_abi != MICROBLAZE_ABI_AUTO)
    microblaze_abi = wanted_abi;

  /* If there is already a candidate, use it.  */
  arches = gdbarch_list_lookup_by_info (arches, &info);
  if ((arches != NULL) && (microblaze_abi != MICROBLAZE_ABI_M64))
    return arches->gdbarch;

  if (microblaze_abi == MICROBLAZE_ABI_M64)
    {
          tdesc = tdesc_microblaze64;
          reg_size = 8;
    }
  if (tdesc == NULL)
    {
      if ((info.bfd_arch_info->mach == bfd_mach_microblaze64) || (microblaze_abi == MICROBLAZE_ABI_M64)) 
        {
    	  tdesc = tdesc_microblaze64;
          reg_size = 8;
   	}
      else
    	tdesc = tdesc_microblaze;
    }
  /* Check any target description for validity.  */
  if (tdesc_has_registers (tdesc))
    {
      const struct tdesc_feature *feature;
      int valid_p;
      int i;

      if ((info.bfd_arch_info->mach == bfd_mach_microblaze64) || (microblaze_abi == MICROBLAZE_ABI_M64))
        feature = tdesc_find_feature (tdesc,
                                    "org.gnu.gdb.microblaze64.core");
      else
        feature = tdesc_find_feature (tdesc,
                                    "org.gnu.gdb.microblaze.core");
      if (feature == NULL)
	return NULL;
      tdesc_data = tdesc_data_alloc ();

      valid_p = 1;
      for (i = 0; i < MICROBLAZE_NUM_REGS; i++)
        valid_p &= tdesc_numbered_register (feature, tdesc_data.get(), i,
                                            microblaze_register_names[i]);
      if ((info.bfd_arch_info->mach == bfd_mach_microblaze64) || (microblaze_abi == MICROBLAZE_ABI_M64))
        feature = tdesc_find_feature (tdesc,
                                    "org.gnu.gdb.microblaze64.stack-protect");
      else
        feature = tdesc_find_feature (tdesc,
                                    "org.gnu.gdb.microblaze.stack-protect");
      if (feature != NULL)
        {
          valid_p = 1;
          valid_p &= tdesc_numbered_register (feature, tdesc_data.get(),
                                              MICROBLAZE_SLR_REGNUM,
                                              "slr");
          valid_p &= tdesc_numbered_register (feature, tdesc_data.get(),
                                              MICROBLAZE_SHR_REGNUM,
                                              "shr");
        }

      if (!valid_p)
        {
         // tdesc_data_cleanup (tdesc_data.get ());
          return NULL;
        }
    }

  /* Allocate space for the new architecture.  */
  microblaze_gdbarch_tdep *tdep = new microblaze_gdbarch_tdep;
  gdbarch = gdbarch_alloc (&info, tdep);

  tdep->gregset = NULL;
  tdep->sizeof_gregset = 0;
  tdep->fpregset = NULL;
  tdep->sizeof_fpregset = 0;
  set_gdbarch_long_double_bit (gdbarch, 128);

  set_gdbarch_num_regs (gdbarch, MICROBLAZE_NUM_REGS);
  set_gdbarch_register_name (gdbarch, microblaze_register_name);
  set_gdbarch_register_type (gdbarch, microblaze_register_type);

  /* Register numbers of various important registers.  */
  set_gdbarch_sp_regnum (gdbarch, MICROBLAZE_SP_REGNUM); 
  set_gdbarch_pc_regnum (gdbarch, MICROBLAZE_PC_REGNUM); 
  
  /* Register set.  
  make_regs (gdbarch); */
  switch (info.bfd_arch_info->mach)
    {
    case bfd_mach_microblaze64:
      set_gdbarch_ptr_bit (gdbarch, 64);
    break;
    }
  if(microblaze_abi == MICROBLAZE_ABI_M64)
    set_gdbarch_ptr_bit (gdbarch, 64);
  
  /* Map Dwarf2 registers to GDB registers.  */
  set_gdbarch_dwarf2_reg_to_regnum (gdbarch, microblaze_dwarf2_reg_to_regnum);

  /* Call dummy code.  */
  set_gdbarch_call_dummy_location (gdbarch, ON_STACK);

  set_gdbarch_return_value (gdbarch, microblaze_return_value);
  set_gdbarch_stabs_argument_has_addr
    (gdbarch, microblaze_stabs_argument_has_addr);

  set_gdbarch_skip_prologue (gdbarch, microblaze_skip_prologue);

  /* Stack grows downward.  */
  set_gdbarch_inner_than (gdbarch, core_addr_lessthan);

  set_gdbarch_breakpoint_kind_from_pc (gdbarch,
				       microblaze_breakpoint::kind_from_pc);
  set_gdbarch_sw_breakpoint_from_kind (gdbarch,
				       microblaze_breakpoint::bp_from_kind);
//  set_gdbarch_memory_remove_breakpoint (gdbarch, microblaze_linux_memory_remove_breakpoint);

//  set_gdbarch_software_single_step (gdbarch, microblaze_software_single_step);

  set_gdbarch_software_single_step (gdbarch, microblaze_software_single_step);

  set_gdbarch_frame_args_skip (gdbarch, 8);

  set_gdbarch_unwind_pc (gdbarch, microblaze_unwind_pc);

  //microblaze_register_g_packet_guesses (gdbarch);

  frame_base_set_default (gdbarch, &microblaze_frame_base);

  /* Hook in ABI-specific overrides, if they have been registered.  */
  gdbarch_init_osabi (info, gdbarch);

  /* Unwind the frame.  */
  dwarf2_append_unwinders (gdbarch);
  frame_unwind_append_unwinder (gdbarch, &microblaze_frame_unwind);
  frame_base_append_sniffer (gdbarch, dwarf2_frame_base_sniffer);
  if (tdesc_data != NULL)
    tdesc_use_registers (gdbarch, tdesc, std::move (tdesc_data));
  //frame_base_append_sniffer (gdbarch, microblaze_frame_sniffer);

  /* If we have register sets, enable the generic core file support.  
  if (tdep->gregset) {
    set_gdbarch_iterate_over_regset_sections (gdbarch,
                                          microblaze_iterate_over_regset_sections);
  }*/
  return gdbarch;
}

void _initialize_microblaze_tdep ();
void
_initialize_microblaze_tdep ()
{
  register_gdbarch_init (bfd_arch_microblaze, microblaze_gdbarch_init);
// static struct cmd_list_element *setmicroblazecmdlist = NULL;
// static struct cmd_list_element *showmicroblazecmdlist = NULL;

  /* Add root prefix command for all "set microblaze"/"show microblaze" commands.  */

  add_setshow_prefix_cmd ("microblaze", no_class,
			 _("Various microblaze specific commands."),
			 _("Various microblaze specific commands."),
			 &setmicroblazecmdlist,&showmicroblazecmdlist,
			 &setlist,&showlist);
#if 0
  add_prefix_cmd ("microblaze", no_class, set_microblaze_command,
                  _("Various microblaze specific commands."),
                  &setmicroblazecmdlist, "set microblaze ", 0, &setlist);

  add_prefix_cmd ("microblaze", no_class, show_microblaze_command,
                  _("Various microblaze specific commands."),
                  &showmicroblazecmdlist, "show microblaze ", 0, &showlist);
#endif

  /* Allow the user to override the ABI.  */
  add_setshow_enum_cmd ("abi", class_obscure, microblaze_abi_strings,
                        &microblaze_abi_string, _("\
Set the microblaze ABI used by this program."), _("\
Show the microblaze ABI used by this program."), _("\
This option can be set to one of:\n\
  auto  - the default ABI associated with the current binary\n\
  m64"),
                       microblaze_abi_update,
		       show_microblaze_abi,
                       &setmicroblazecmdlist, &showmicroblazecmdlist);
  
  initialize_tdesc_microblaze_with_stack_protect ();
  initialize_tdesc_microblaze ();
  initialize_tdesc_microblaze64_with_stack_protect ();
  initialize_tdesc_microblaze64 ();
  /* Debug this files internals.  */
  add_setshow_zuinteger_cmd ("microblaze", class_maintenance,
			     &microblaze_debug_flag, _("\
Set microblaze debugging."), _("\
Show microblaze debugging."), _("\
When non-zero, microblaze specific debugging is enabled."),
			     NULL,
			     NULL,
			     &setdebuglist, &showdebuglist);

}
